/*
 * Copyright (C) 2011 Benjamin Franzke
 * Copyright (C) 2010 Intel Corporation
 *
 * Permission to use, copy, modify, distribute, and sell this software and its
 * documentation for any purpose is hereby granted without fee, provided that
 * the above copyright notice appear in all copies and that both that copyright
 * notice and this permission notice appear in supporting documentation, and
 * that the name of the copyright holders not be used in advertising or
 * publicity pertaining to distribution of the software without specific,
 * written prior permission.  The copyright holders make no representations
 * about the suitability of this software for any purpose.  It is provided "as
 * is" without express or implied warranty.
 *
 * THE COPYRIGHT HOLDERS DISCLAIM ALL WARRANTIES WITH REGARD TO THIS SOFTWARE,
 * INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO
 * EVENT SHALL THE COPYRIGHT HOLDERS BE LIABLE FOR ANY SPECIAL, INDIRECT OR
 * CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE,
 * DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER
 * TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE
 * OF THIS SOFTWARE.
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <stdbool.h>
#include <assert.h>
#include <unistd.h>
#include <sys/mman.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/statvfs.h>
#include <fcntl.h>
#include <errno.h>
#include <sys/epoll.h>
#include <sys/stat.h>
#include <strings.h>

#include "VideoPlayer_ilmUtil.h"
#include "wayland_backend.h"

#include <wayland-client.h>
#include "VideoPlayer_shm.h"
#include "dispvidctrl_AppMain_Trace.h"

#ifdef VARIANT_S_FTR_WESTON_FOR_GEN4
#include "early_ilm_controller.h"
#endif //VARIANT_S_FTR_WESTON_FOR_GEN4

#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#define ET_TRACE_INFO_ON
#include "etrace_if.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_DISPVIDCTRL_CLIENT_VIDEOPLAYER
#include "trcGenProj/Header/VideoPlayer_shm.c.trc.h"
#endif

#ifndef UNUSED
#define UNUSED(a) ((a) = (a))
#endif

#ifndef ARRAY_LENGTH
#define ARRAY_LENGTH(a) (sizeof (a) / sizeof (a)[0])
#endif

#define SERVERINFO_GET_CONNECTION_ID    0

#define NUM_BUFFERS 10

// some useful defines and macros to directly draw pixels of different colors to a buffer

// - On the iMX, the R(ed),G(reen),B(lue) and A(lpha) bytes are stored in memory in B,G,R,A order (Alpha: 0x00=transparent, 0xFF=opaque).
// - The iMX is a LittleEndian machine, so for dword-wise access, the four R,G,B,A bytes of one pixel correspond to the 32bit-value 0xAARRGGBB.
// With that, we define the (unsigned) 32-bit values for the colors we need:
#define U32_RGBA_BLACK  0xFF000000u
#define U32_RGBA_BLUE   0xFF0000FFu  // For CVBS it is required to show a blue screen when there is no CVBS signal
#define U32_RGBA_GREEN  0xFF000000u  // NCG3D-159976 setting to black screen based on customer request // For LVDS it is required to show a green screen when there is no LVDS signal
#define U32_RGBA_TRANSP 0x00000000u
#define U32_RGBA_PINK   0xFFFF00FFu  // piggy-pink, e.g. for testing
#define U32_RGBA_GRAY   0xFF808080u  // Gray untill the video signal gets right, basically to avoid the screen being whatsoever.

#define VP_SHM_PUT_PIXELS(pu32,npix,u32rgba)     for (int i=(npix); i>0; --i, ++(pu32)) { *(pu32) = (u32rgba); }

#define VP_SHM_PUT_1COLOR_ROWS(pu32,nrows,u32rgba) \
   for (int j=(nrows); j>0; --j) { \
      VP_SHM_PUT_PIXELS((pu32),imgwidth,(u32rgba)) \
   }
#define VP_SHM_PUT_2COLOR_ROWS(pu32,nrows, npix1,u32rgba1, npix2,u32rgba2) \
   for (int j=(nrows); j>0; --j) { \
      VP_SHM_PUT_PIXELS((pu32),(npix1),(u32rgba1)) \
      VP_SHM_PUT_PIXELS((pu32),(npix2),(u32rgba2)) \
   }
#define VP_SHM_PUT_3COLOR_ROWS(pu32,nrows, npix1,u32rgba1, npix2,u32rgba2, npix3,u32rgba3) \
   for (int j=(nrows); j>0; --j) { \
      VP_SHM_PUT_PIXELS((pu32),(npix1),(u32rgba1)) \
      VP_SHM_PUT_PIXELS((pu32),(npix2),(u32rgba2)) \
      VP_SHM_PUT_PIXELS((pu32),(npix3),(u32rgba3)) \
   }

#define VP_SHM_WINSIZE_IS(expwdth,exphght) ((imgwidth == (expwdth)) && (imgheight == (exphght)))

#define EN_SURFACE_VISIBILITY_SET 1
#define EN_SURFACE_DESTROYED 2


int ret = 0;
unsigned int graphics_layerid;
unsigned int rvc_layerid;
unsigned int graphics_surfaceid;
unsigned int rvc_surfaceid;
unsigned int window_height;
unsigned int window_width;
char* guideline_buffer;

// redrawGuidelinesCurrent and enBlackScreenTypeRequestCurrent relate to a draw that is in progress
int redrawGuidelinesCurrent = 0;
enum ten_VideoPlayer_BlackScreenType enBlackScreenTypeRequestCurrent = EN_VIDEOPLAYER__BLACKSCREENTYPE_NONE;
// redrawGuidelinesPending and enBlackScreenTypeRequestPending relate to a draw that has been requested but has yet to start
int redrawGuidelinesPending = 0;
enum ten_VideoPlayer_BlackScreenType enBlackScreenTypeRequestPending = EN_VIDEOPLAYER__BLACKSCREENTYPE_NONE;

bool green_screen = false;
bool blue_screen = false;
bool video_signal_error = false;
bool force_Blackscreen = false;
bool stop_guideline_visibility = false;
bool bSleepLong = false;

bool hfp_tee_active = false;
unsigned int rvc_tee_layerid;
unsigned int rvc_tee_surfaceid;
bool hfp_tee_surfaceactive = false;
bool rvc_active = false;
bool rvc_visible = false;
bool graphics_visible = false;
bool hfp_tee_visible = false;
bool visibility_changed = false;
int surface_visibility = EN_SURFACE_DESTROYED;
long int MemoryAllocated = 0;


#ifdef VARIANT_S_FTR_ENABLE_AIVI_RCTA

struct window *rcta_window = NULL;
struct display *rcta_display = NULL;
char* rcta_image_buffer = NULL;

struct rcta_window_params
{
    unsigned short window_X_pos;
    unsigned short window_Y_pos;
    unsigned short window_width;
    unsigned short window_height;
    unsigned short rcta_layerid;
    unsigned short rcta_surfaceid;
}rcta_window_size;

bool rcta_graphics_ready = false;
bool rcta_init_done = false;
bool stop_rcta_graphic = false;

#endif //VARIANT_S_FTR_ENABLE_AIVI_RCTA


#ifdef VARIANT_S_FTR_ENABLE_BUMPER_ICON_P42R//CRQ 844 P42R Bumper Icon
struct window *bumper_window = NULL;
struct display *bumper_display = NULL;
char* bumper_image_buffer = NULL;

struct bumper_window_params
{
    unsigned short window_X_pos;
    unsigned short window_Y_pos;
    unsigned short window_width;
    unsigned short window_height;
    unsigned short bumper_layerid;
    unsigned short bumper_surfaceid;
};

struct bumper_window_params *bumper_window_size = NULL;

bool bumper_graphics_ready = false;
bool bumper_init_done = false;
bool stop_bumper_graphic = false;
bool CreateBumperSurface = false;
#endif//VARIANT_S_FTR_ENABLE_BUMPER_ICON_P42R

#if defined(GEN3X86) || defined (GEN4LSIM)
//For LSIM this is always true
bool shm_graphics_ready = true;
#else //GEN3X86 GEN4LSIM
bool shm_graphics_ready = false;
#endif //GEN3X86 GEN4LSIM

int running = 0;
int stopped = 0;
int display_control = 0;

#ifdef VARIANT_S_FTR_WESTON_FOR_GEN4
//To support setting of Gamma and CSC required by Dimming and supported by Wayland Backend
int redrawGammaPending              = 0;
int redrawCSCPending                = 0;
double dRGamma                      = 1;
double dGGamma                      = 1;
double dBGamma                      = 1;
unsigned int u16CSCHue              = 0;
unsigned int u16CSCSaturation       = 100;
unsigned int u16CSCBrightness       = 100;
unsigned int u16CSCContrast         = 105;
unsigned int u16CSCHueOffset        = 0;
unsigned int u16CSCSaturationOffset = 0;
unsigned int u16CSCBrightnessOffset = 1;
#endif //VARIANT_S_FTR_WESTON_FOR_GEN4

//#######################################################################################################################################################################################################
//#######################################################################################################################################################################################################
// FUNCTIONS CONTAINING CODE FROM ORIGINAL ADIT SHM DELIVERY 
//#######################################################################################################################################################################################################
//#######################################################################################################################################################################################################

#ifndef VARIANT_S_FTR_WESTON_FOR_GEN4

// shm is used for gen3. cairo is used for gen4

static const struct wl_interface *types[] = {
   NULL,
};

static const struct wl_message serverinfo_requests[] = {
   { "get_connection_id", "", types + 0 },
};

static const struct wl_message serverinfo_events[] = {
   { "connection_id", "u", types + 0 },
};

const struct wl_interface serverinfo_interface = {
   "serverinfo", 1,
   ARRAY_LENGTH(serverinfo_requests), serverinfo_requests,
   ARRAY_LENGTH(serverinfo_events), serverinfo_events,
};


struct display {
   struct wl_display *display;
   struct wl_registry *registry;
   struct wl_compositor *compositor;
   struct wl_shm *shm;
   struct serverinfo* wl_ext_serverinfo;
   unsigned int wl_connect_id;
   uint32_t formats;
};

struct buffer {
   struct wl_buffer *buffer;
   void *shm_data;
   int busy;
};

struct window {
   struct display *display;
   int width, height;
   struct wl_surface *surface;
   struct buffer buffers[NUM_BUFFERS];
   struct wl_callback *callback;

   unsigned int surface_id;
   unsigned int layer_id;
};

struct display *display;
struct window *window;

static int
set_cloexec_or_close(int fd)
{
	long flags;

	if (fd == -1)
		return -1;

	flags = fcntl(fd, F_GETFD);
	if (flags == -1)
    {
	   close(fd);
	   return -1;
    }

	if (fcntl(fd, F_SETFD, flags | FD_CLOEXEC) == -1)
    {
	   close(fd);
	   return -1;
    }

	return fd;

// err:
// 	close(fd);
// 	return -1;
}


static int
create_tmpfile_cloexec(char *tmpname)
{
	int fd;

#ifdef HAVE_MKOSTEMP
	fd = mkostemp(tmpname, O_CLOEXEC);
	if (fd >= 0)
		unlink(tmpname);
#else //HAVE_MKOSTEMP
	fd = mkstemp(tmpname);
	if (fd >= 0) {
		fd = set_cloexec_or_close(fd);
		unlink(tmpname);
	}
#endif

	return fd;
}

/*
* this function is to check the available memory inside /tmp 
* function statvfs() returns the contents insize tmp folder in data structure variable
* AS Data.f_bsize is 4K hence our required size (which is in bytes) is devided by 4K to compare

*/
//  struct statvfs {
//      unsigned long  f_bsize;    /* Filesystem block size */
//      unsigned long  f_frsize;   /* Fragment size */
//      fsblkcnt_t     f_blocks;   /* Size of fs in f_frsize units */
//      fsblkcnt_t     f_bfree;    /* Number of free blocks */
//      fsblkcnt_t     f_bavail;   /* Number of free blocks for
//                                    unprivileged users */
//      fsfilcnt_t     f_files;    /* Number of inodes */
//      fsfilcnt_t     f_ffree;    /* Number of free inodes */
//      fsfilcnt_t     f_favail;   /* Number of free inodes for
//                                    unprivileged users */
//      unsigned long  f_fsid;     /* Filesystem ID */
//      unsigned long  f_flag;     /* Mount flags */
//      unsigned long  f_namemax;  /* Maximum filename length */
    
void CheckAvailableMemory(off_t size)
{
   struct statvfs Data;
   char filePath[20] = "/tmp";
   int RbufferinBytes = size;

   if((statvfs(filePath,&Data)) < 0 ) {
         ETG_TRACE_FATAL((" CheckAvailableMemory :: Failed to stat %s:", filePath));
   } 
   else {
        ETG_TRACE_FATAL(("CheckAvailableMemory::RbufferinBytes %i", RbufferinBytes));
        ETG_TRACE_FATAL(("CheckAvailableMemory::Disk %s: ", filePath));
        ETG_TRACE_FATAL(("CheckAvailableMemory::block size: %u", Data.f_bsize));
        ETG_TRACE_FATAL(("CheckAvailableMemory::total no blocks: %i", Data.f_blocks));
        ETG_TRACE_FATAL(("CheckAvailableMemory::free blocks: %i", Data.f_bavail));
		   if (RbufferinBytes > (Data.f_bavail*Data.f_bsize) )
		   {
               ETG_TRACE_FATAL(("CheckAvailableMemory::RbufferinBytes %d  and free blocks: %d", RbufferinBytes,  Data.f_bavail));
               ETG_TRACE_ERRMEM(("CheckAvailableMemory::RbufferinBytes %d  and free blocks: %d", RbufferinBytes, Data.f_bavail));
		   }	   
   }

}


/*
 * Create a new, unique, anonymous file of the given size, and
 * return the file descriptor for it. The file descriptor is set
 * CLOEXEC. The file is immediately suitable for mmap()'ing
 * the given size at offset zero.
 *
 * The file should not have a permanent backing store like a disk,
 * but may have if XDG_RUNTIME_DIR is not properly implemented in OS.
 *
 * The file name is deleted from the file system.
 *
 * The file is suitable for buffer sharing between processes by
 * transmitting the file descriptor over Unix sockets using the
 * SCM_RIGHTS methods.
 */
int
os_create_anonymous_file(off_t size)
{
	
	// before creating a file we need to check for avaialble memory inside our temp folder 
	CheckAvailableMemory(size);
	
	static const char template[] = "/weston-shared-XXXXXX";
	const char *path;
	char *name;
	int fd;

	path = getenv("XDG_RUNTIME_DIR");
	if (!path) {
		errno = ENOENT;
        ETG_TRACE_FATAL((" create_shm_buffer of file path %d ", errno));
		return -1;
	}

	name = malloc(strlen(path) + sizeof(template));
	if (!name)
		return -1;

	strcpy(name, path);
	strcat(name, template);

	fd = create_tmpfile_cloexec(name);
    ETG_TRACE_FATAL((" create_shm_buffer of file path %s ", name));


	free(name);

	if (fd < 0)
		return -1;

	if (ftruncate(fd, size) < 0) {
		close(fd);
		return -1;
	}

	return fd;
}

static void serverinfo_cb_impl(void *data, struct serverinfo *pServerinfo, unsigned int client_handle)
{
   UNUSED(pServerinfo);

   struct display * disp = (struct display *)data;
   disp->wl_connect_id = client_handle;
}

struct serverinfo_listener {
   void (*connection_id)(void *data,
                         struct serverinfo *serverinfo,
                         uint32_t);
}; //lint !e754

struct serverinfo_listener serverinfo_cb = {
   serverinfo_cb_impl
};

static void
buffer_release(void *data, struct wl_buffer *buffer)
{
   struct buffer *mybuf = data;
   UNUSED(buffer);
   mybuf->busy = 0;
}

static const struct wl_buffer_listener buffer_listener = {
   buffer_release
};

static int
create_shm_buffer(struct display *display, struct buffer *buffer,
                  int width, int height, uint32_t format)
{ //lint !e578
   struct wl_shm_pool *pool;
   int fd, size, stride;
   void *data;

   stride = width * 4;
   size = stride * height;

   ETG_TRACE_FATAL(("[%d bytes] create_shm_buffer of file size", size));
   fd = os_create_anonymous_file(size);
   if (fd < 0) {
      ETG_TRACE_FATAL( ( "[%d ms] dispvidctrl shm create_shm_buffer() creating a buffer file for %d failed", OSAL_ClockGetElapsedTime(), size));
      return -1;
   }

   /*pointer to the buffer memory which will be displayed*/
   data = mmap(NULL, size, PROT_READ | PROT_WRITE, MAP_SHARED, fd, 0);
   if (data == MAP_FAILED) {
      ETG_TRACE_FATAL( ( "[%d ms] dispvidctrl shm create_shm_buffer() mmap failed", OSAL_ClockGetElapsedTime()));

      close(fd);
      return -1;
   }
   else{

	struct stat st; /*declare stat variable*/
	/*get the size using stat()*/

	 /* find size of input file */
	 if (fstat (fd,&st) == 0)
	 {
         MemoryAllocated = st.st_size;
		
	 }else
	 {
         // MemoryAllocated = size;
         ETG_TRACE_ERRMEM(("create_shm_buffer fstat returned -1 errno[%s]",strerror(errno) ))
		 close(fd);
         munmap(data,size);
         return -1;
     }
     ETG_TRACE_ERRMEM(("create_shm_buffer size[%d bytes] MemoryAllocated[%d bytes]", size,MemoryAllocated));
   }

   if(MemoryAllocated != size)
   {
       ETG_TRACE_ERRMEM(("[%d bytes] MemoryAllocated is not equal to actal size [%d bytes]", MemoryAllocated, size));
	   close(fd);	   
       munmap(data,size);
       return -1;
   }
   
   ETG_TRACE_FATAL( ( "[%d ms] dispvidctrl shm create_shm_buffer() size of meory allocated = [%d]", OSAL_ClockGetElapsedTime(), MemoryAllocated));   

   pool = wl_shm_create_pool(display->shm, fd, size);
   buffer->buffer = wl_shm_pool_create_buffer(pool, 0,
                     width, height,
                     stride, format);
   wl_buffer_add_listener(buffer->buffer, &buffer_listener, buffer);
   wl_shm_pool_destroy(pool);
   close(fd);

   buffer->shm_data = data;

   return 0;
}

static struct window *
create_window(struct display *display, int width, int height)
{//lint !e578
   struct window *window; //lint !e578

   window = calloc(1, sizeof *window);
   if (!window)
      return NULL;

   window->callback = NULL;
   window->display = display;
   window->width = width;
   window->height = height;
   window->surface = wl_compositor_create_surface(display->compositor);

   window->layer_id = 0xFFFFFFFF;
   window->surface_id = 0xFFFFFFFF;

   return window;
}

static void
destroy_window(struct window *window)
{//lint !e578
    int i = 0;
   if (window->callback)
      wl_callback_destroy(window->callback);

   for (i = 0; i < NUM_BUFFERS; i++)
   {
        if (window->buffers[i].buffer)
            wl_buffer_destroy(window->buffers[i].buffer);
   }
   wl_surface_destroy(window->surface);
   free(window);
}

static struct buffer *
window_next_buffer(struct window *window)
{//lint !e578
   struct buffer *buffer = NULL;
   int ret = 0;//lint !e578
   int cur_buf = 0;

   if (!window) {
       return NULL;
   }

   for (cur_buf = 0; cur_buf < NUM_BUFFERS; cur_buf++)
   {
        if (!window->buffers[cur_buf].busy)
        {
            buffer = &window->buffers[cur_buf];
            break;
        }
   }
   if (buffer == NULL)
   {
       /*all buffers are busy, no free buffer available*/
       return NULL;
   }

   if (!buffer->buffer) {
       /*here the memory is actually allocated
        * only ones, in the first run*/
      ret = create_shm_buffer(window->display, buffer,
               window->width, window->height,
               WL_SHM_FORMAT_ARGB8888);

      if (ret < 0)
         return NULL;

      /* Check parameters before calling memset
         Note - it is not possible to check whether we have a valid heap address or whether the size goes beyond the end of the heap
                This is NCG3D-106485 and later NCG3D-146915 */
	
#if 0
      // Old code
      if ( (window->width != window_width) || (window->height != window_height) || (buffer->shm_data == 0) ) {
         ETG_TRACE_FATAL( ( "[%d ms] dispvidctrl shm window_next_buffer() - heap corruption - ignoring memset - window->width %d [%d] window->height %d [%d] buffer->shm_data [%d]", 
             OSAL_ClockGetElapsedTime(), window->width, window_width, window->height, window_height, buffer->shm_data));
      } else {
         memset(buffer->shm_data, 0xff,
                window->width * window->height * 4);
      }
#endif //0

      // eis2hi: Coverity might complain about "condition '(window != 0)' always true" here. If it does, see my comment a few lines further down
      if ( (window != 0) && (window->width == window_width) && (window->height == window_height) && (buffer != 0) && (buffer->shm_data != 0) ) {
         bzero(buffer->shm_data, MemoryAllocated);
      } else {
         // eis2hi: Coverity might complain about "logically dead code" and/or "condition always true" here. If it does, this is intentional, because
         //         under *normal* circumstances, 'window != 0' *should* always be true, and this else branch *should* be unreachable.
         //         However, had cases somoe strange cases in the past where at this point, window and/or buffer suddenly *was* NULL here,
         //         and this else branch served to get more information about this situation
         ETG_TRACE_FATAL( ( "[%d ms] dispvidctrl shm window_next_buffer() - heap corruption - ignoring memset - window [0x%X] buffer [0x%X]", OSAL_ClockGetElapsedTime(), window, buffer));
         if (window != 0) {
            ETG_TRACE_FATAL( ( "[%d ms] dispvidctrl shm window_next_buffer() - heap corruption - ignoring memset - window->width %d [%d] window->height %d [%d]",
                OSAL_ClockGetElapsedTime(), window->width, window_width, window->height, window_height ));
         }
         if (buffer != 0) {
            ETG_TRACE_FATAL( ( "[%d ms] dispvidctrl shm window_next_buffer() - heap corruption - ignoring memset - buffer->shm_data [%d]", OSAL_ClockGetElapsedTime(), buffer->shm_data));
         }
         return NULL;
      }
   }

   return buffer;
}

static void my_frame_callback(void *data, struct wl_callback *callback, uint32_t time)
{//lint !e578
    UNUSED(data);
    UNUSED(callback);
    UNUSED(time);
    ETG_TRACE_FATAL( ( "[%d ms] dispvidctrl shm my_frame_callback() - frame processed", OSAL_ClockGetElapsedTime()));
}

static const struct wl_callback_listener frame_listener = {
   my_frame_callback
};

static void
shm_format(void *data, struct wl_shm *wl_shm, uint32_t format)
{
   struct display *d = data;
   UNUSED(wl_shm);
   d->formats |= (1 << format);
}

struct wl_shm_listener shm_listenter = {
   shm_format
};

static void
registry_handle_global(void *data, struct wl_registry *registry,
             uint32_t id, const char *interface, uint32_t version)
{
   struct display *d = data;
   UNUSED(version);

   if (strcmp(interface, "wl_compositor") == 0)
   {
      d->compositor =
         wl_registry_bind(registry,
                id, &wl_compositor_interface, 1);
   }
   else if (strcmp(interface, "wl_shm") == 0)
   {
      d->shm = wl_registry_bind(registry,
                 id, &wl_shm_interface, 1);
      wl_shm_add_listener(d->shm, &shm_listenter, d);
   }
   else if (!strcmp(interface, "serverinfo"))
   {
        d->wl_ext_serverinfo = (struct serverinfo*)wl_registry_bind( registry, id, &serverinfo_interface, 1);

        wl_proxy_add_listener((struct wl_proxy *) d->wl_ext_serverinfo,
                             (void (**)(void)) &serverinfo_cb, data);

        wl_proxy_marshal((struct wl_proxy *) d->wl_ext_serverinfo,
                         SERVERINFO_GET_CONNECTION_ID);
    }
}

static void
registry_handle_global_remove(void *data, struct wl_registry *registry,
               uint32_t name)
{
    UNUSED(data);
    UNUSED(registry);
    UNUSED(name);
}

static const struct wl_registry_listener registry_listener = {
   registry_handle_global,
   registry_handle_global_remove
};

static struct display *
create_display(void)
{
   struct display *display; //lint !e578

   display = malloc(sizeof *display);
   if(display != 0)
   {
      display->display = wl_display_connect(NULL);
      assert(display->display);

      display->formats = 0;
      display->registry = wl_display_get_registry(display->display);
      wl_registry_add_listener(display->registry, &registry_listener, display);
      wl_display_roundtrip(display->display);
      if (display->shm == NULL) {
         ETG_TRACE_FATAL( ( "[%d ms] dispvidctrl shm create_display() No wl_shm global", OSAL_ClockGetElapsedTime()));
         exit(1);
      }

      wl_display_roundtrip(display->display);

      if (!(display->formats & (1 << WL_SHM_FORMAT_XRGB8888))) {
         ETG_TRACE_FATAL( ( "[%d ms] dispvidctrl shm create_display() WL_SHM_FORMAT_XRGB32 not available", OSAL_ClockGetElapsedTime()));
         exit(1);
      }
   
      return display;
   }
   return 0;
}

static void
destroy_display(struct display *display)
{//lint !e578
   if (display->shm)
      wl_shm_destroy(display->shm);

   if (display->compositor)
      wl_compositor_destroy(display->compositor);

   wl_registry_destroy(display->registry);
   wl_display_flush(display->display);
   wl_display_disconnect(display->display);
   free(display);
}

static void init_lm_client(struct window* w, uint layer_id, uint surface_id)
{
    if(NULL == getenv("XDG_RUNTIME_DIR"))
    {
      setenv("XDG_RUNTIME_DIR","/tmp/", 1);
    }

    ETG_TRACE_FATAL( ( "[%d ms] dispvidctrl shm init_lm_client() CREATE ON LAYERID: %d ,SURFACEID :%d", OSAL_ClockGetElapsedTime(), layer_id, surface_id));

    w->layer_id = layer_id;
    w->surface_id = surface_id;

    struct wl_proxy* pxy = (struct wl_proxy*)w->surface;
	unsigned long native_ilm_handle = (w->display->wl_connect_id << 16) | (uint32_t) wl_proxy_get_id(pxy);
	
#ifndef VARIANT_S_FTR_WESTON_FOR_GEN4
	vSurfaceCreate(native_ilm_handle,
                   w->width,
                   w->height,
                   ILM_PIXELFORMAT_RGBA_8888,
                   w->surface_id,
				   COMMIT_DELAYED);


    vUpdateInputEventAcceptanceOn(w->surface_id, COMMIT_DELAYED);
#endif //VARIANT_S_FTR_WESTON_FOR_GEN4

    vSurfaceSetDestinationRectangle(w->surface_id, 0, 0, w->width, w->height, COMMIT_DELAYED);
    vSurfaceSetSourceRectangle(w->surface_id, 0, 0, w->width, w->height, COMMIT_DELAYED);

    // Don't set visible until draw complete
    vSurfaceSetVisibility(w->surface_id, VISIBILITY_OFF, COMMIT_DELAYED);

    ETG_TRACE_FATAL (("[%d ms] dispvidctrl shm init_lm_client() - LayerManagerControl Add Surface %d to Layer %d", OSAL_ClockGetElapsedTime(), surface_id, layer_id));
    bLayerAddSurface(w->layer_id, w->surface_id, COMMIT_DELAYED);

    vCommitChanges_util(); //lint !e746
}

static void deinit_lm_client(struct window* w)
{
    wl_display_flush( w->display->display );
    wl_display_roundtrip( w->display->display);
	
#ifndef VARIANT_S_FTR_WESTON_FOR_GEN4
    vSurfaceRemove(w->surface_id, COMMIT_NOW);
#endif //VARIANT_S_FTR_WESTON_FOR_GEN4
}

static void redraw(void *data)
{
   struct window *window = data; //lint !e578
   struct buffer *buffer;
   tU8 u8RetryCount = 0;

   if(redrawGuidelinesPending == 1)
   {
      ETG_TRACE_FATAL( ( "[%d ms] dispvidctrl shm redraw() - redrawGuidelinesPending [%d] - update surface", OSAL_ClockGetElapsedTime(), redrawGuidelinesPending));
      redrawGuidelinesPending = 0;
      redrawGuidelinesCurrent = 1;
	  enBlackScreenTypeRequestCurrent = enBlackScreenTypeRequestPending;

      buffer = window_next_buffer(window);
      if (!buffer) 
      {
         ETG_TRACE_FATAL( ( "[%d ms] dispvidctrl shm redraw() Failed to create the first buffer or Both buffers busy at redraw(). Server bug?", OSAL_ClockGetElapsedTime()));
         if(u8RetryCount == 0)
         {
            u8RetryCount = 1;
            buffer = window_next_buffer(window);		  
         }
         if(!buffer)
         { 
            ETG_TRACE_FATAL( ( "[%d ms] dispvidctrl shm redraw() Failed after retrying create buffer, triggering reset", OSAL_ClockGetElapsedTime()));
            ETG_TRACE_ERRMEM( ( "dispvidctrl shm redraw() Failed after retrying create buffer, triggering reset"));
            abort();
         }
      }

      draw_guidelines((void*)buffer->shm_data);

      if (window->callback)
      {
         wl_callback_destroy(window->callback);
      }

      window->callback = wl_surface_frame(window->surface);
      wl_callback_add_listener(window->callback, &frame_listener, window);

      wl_surface_attach(window->surface, buffer->buffer, 0, 0);
      wl_surface_damage(window->surface, 0, 0, window->width, window->height);
      wl_surface_commit(window->surface);
      buffer->busy = 1;
   }
   else
   {
      ETG_TRACE_FATAL( ( "[%d ms] dispvidctrl shm redraw() - redrawGuidelinesPending [0] - nothing to do", OSAL_ClockGetElapsedTime()));
   }
}

#endif //VARIANT_S_FTR_WESTON_FOR_GEN4

//#######################################################################################################################################################################################################
//#######################################################################################################################################################################################################
// FUNCTIONS CONTAINING BOTH DRAW AND DISPLAY 
//#######################################################################################################################################################################################################
//#######################################################################################################################################################################################################

#ifdef VARIANT_S_FTR_WESTON_FOR_GEN4

void vHandleVisibilityOnOff()
{
    ETG_TRACE_FATAL( ( "[%d ms] dispvidctrl shm vHandleVisibilityOnOff() Entered - enBlackScreenTypeRequestCurrent [%d] stop_guideline_visibility [%d] rvc_active [%d]", OSAL_ClockGetElapsedTime(), enBlackScreenTypeRequestCurrent, stop_guideline_visibility, rvc_active));
	
    //Perform LayerManagerControl commands after surface has been updated
    if(enBlackScreenTypeRequestCurrent != EN_VIDEOPLAYER__BLACKSCREENTYPE_NONE)
    {
        enBlackScreenTypeRequestCurrent = EN_VIDEOPLAYER__BLACKSCREENTYPE_NONE;
        redrawGuidelinesCurrent = 0;

        // make it visible only if RVC has not stopped meanwhile
        if ( stop_guideline_visibility == false )
        {
            ETG_TRACE_FATAL( ( "[%d ms] dispvidctrl shm vHandleVisibilityOnOff() - black screen draw completed - switch on RVC and guideline visibility - ilmUtil", OSAL_ClockGetElapsedTime() ));
            bLayerAddSurface(graphics_layerid, graphics_surfaceid, COMMIT_DELAYED);
            vSurfaceSetVisibility(graphics_surfaceid, VISIBILITY_ON, COMMIT_DELAYED);
            vLayerSetVisibility(graphics_layerid, VISIBILITY_ON, COMMIT_DELAYED);
            vLayerSetVisibility(rvc_layerid, VISIBILITY_ON, COMMIT_DELAYED);
            rvc_visible = true;
            graphics_visible = true;
            visibility_changed = true;

            // make it visible only if hfp dual window is enabled
            if(hfp_tee_active == true)
            {
                // make tee visible only for AVM Maneuver screen and not for AVM settings screen
                // Do not make it visible if it is already visible
                if( (hfp_tee_visible == false) && (hfp_tee_surfaceactive == true) )
                {
                    ETG_TRACE_FATAL( ( "[%d ms] dispvidctrl shm vHandleVisibilityOnOff() - black screen draw completed - switch on tee visibility - ilmUtil", OSAL_ClockGetElapsedTime() ));
                    vSurfaceSetVisibility(rvc_tee_surfaceid, VISIBILITY_ON, COMMIT_DELAYED);
                    vLayerSetVisibility(rvc_tee_layerid, VISIBILITY_ON, COMMIT_DELAYED);
                    hfp_tee_visible = true;
                    visibility_changed = true;
                }
            }

            //Commit the changes only if surface/Layer visibility has changed.
            if(visibility_changed == true)
            {
                vCommitChanges_util();
                visibility_changed = false;
            }
        }
        else
        {
            //stop_guideline_visibility = false;
            ETG_TRACE_FATAL (("[%d ms] dispvidctrl shm vHandleVisibilityOnOff() - RVC has stopped meanwhile, don't make the Layers visible", OSAL_ClockGetElapsedTime()));
        }
    }
    else
    {
        // make it visible only if RVC has not stopped meanwhile
        if ( (stop_guideline_visibility == false) && (rvc_active == true) )
        {
            if (video_signal_error == false)
            {
                // No video signal error. Only switch on visibility if not already visible
                if ( (rvc_visible == false) || (graphics_visible == false) )
                {
                    // There is no current video signal error so we can switch on both video and guidelines
                    ETG_TRACE_FATAL( ( "[%d ms] dispvidctrl shm vHandleVisibilityOnOff() - switch on RVC and guideline visibility - ilmUtil", OSAL_ClockGetElapsedTime() ));
                    bLayerAddSurface(graphics_layerid, graphics_surfaceid, COMMIT_DELAYED);
                    vSurfaceSetVisibility(graphics_surfaceid, VISIBILITY_ON, COMMIT_DELAYED);
                    vLayerSetVisibility(graphics_layerid, VISIBILITY_ON, COMMIT_DELAYED);
                    vLayerSetVisibility(rvc_layerid, VISIBILITY_ON, COMMIT_DELAYED);
                    rvc_visible = true;
                    graphics_visible = true;
                    visibility_changed = true;
                }
                // make it visible only if hfp dual window is enabled
                if(hfp_tee_active == true)
                {
                    // make tee visible only for AVM Maneuver screen and not for AVM settings screen
                    // Do not make it visible if it is already visible
                    if( (hfp_tee_visible == false) && (hfp_tee_surfaceactive == true) )
                    {
                        ETG_TRACE_FATAL( ( "[%d ms] dispvidctrl shm vHandleVisibilityOnOff() - black screen draw completed - switch on tee visibility - ilmUtil", OSAL_ClockGetElapsedTime() ));
                        vSurfaceSetVisibility(rvc_tee_surfaceid, VISIBILITY_ON, COMMIT_DELAYED);
                        vLayerSetVisibility(rvc_tee_layerid, VISIBILITY_ON, COMMIT_DELAYED);
                        hfp_tee_visible = true;
                        visibility_changed = true;
                    }
                }
             
                //Commit the changes only if surface/Layer visibility has changed.
                if(visibility_changed == true)
                {
                    vCommitChanges_util();
                    visibility_changed = false;
                }
            }
            else
            {
                // There is currently a video signal error so we can switch on only guidelines
                // Do not switch on if already visible
                if (graphics_visible == false)
                {
                    ETG_TRACE_FATAL( ( "[%d ms] dispvidctrl shm vHandleVisibilityOnOff() - current video signal error - switch on only guideline visibility - ilmUtil", OSAL_ClockGetElapsedTime() ));
                    bLayerAddSurface(graphics_layerid, graphics_surfaceid, COMMIT_DELAYED);
                    vSurfaceSetVisibility(graphics_surfaceid, VISIBILITY_ON, COMMIT_DELAYED);
                    vLayerSetVisibility(graphics_layerid, VISIBILITY_ON, COMMIT_DELAYED);
                    vCommitChanges_util();
                    graphics_visible = true;
                }
            }
        }
        else
        {
            //stop_guideline_visibility = false;
            ETG_TRACE_FATAL (("[%d ms] dispvidctrl shm vHandleVisibilityOnOff() - RVC has stopped (stop_guideline_visibility), don't make the Layers visible", OSAL_ClockGetElapsedTime()));
        }
    }

    while( (redrawGuidelinesPending == 0) && (redrawGammaPending == 0) && (redrawCSCPending == 0) )
    {
         // sleep long if camera is not requested to reduce system load (NCG3D-83954)
        if (bSleepLong == true)
        {
           usleep(10000); // 10ms
        }
        else
        {
           usleep(1000); // 1ms
        }
         
        if ( stop_guideline_visibility == true ) // RVC stopped meanwhile ?
        {
            if ( (rvc_visible == true) || (graphics_visible == true) )
            {
                ETG_TRACE_FATAL( ( "[%d ms] dispvidctrl shm vHandleVisibilityOnOff() - RVC stopping - switch off rvc and guideline visibility - ilmUtil", OSAL_ClockGetElapsedTime() ));
                vLayerRemoveSurface(graphics_layerid, graphics_surfaceid, COMMIT_DELAYED);
                vLayerSetVisibility(graphics_layerid, VISIBILITY_OFF, COMMIT_DELAYED);
                vLayerSetVisibility(rvc_layerid, VISIBILITY_OFF, COMMIT_DELAYED);

                if((hfp_tee_active == true) && (hfp_tee_visible == true) )
                {
                    ETG_TRACE_FATAL( ( "[%d ms] dispvidctrl shm vHandleVisibilityOnOff() - RVC stopping - switch off tee visibility - ilmUtil", OSAL_ClockGetElapsedTime() ));
                    vSurfaceSetVisibility(rvc_tee_surfaceid, VISIBILITY_OFF, COMMIT_DELAYED);
                    vLayerSetVisibility(rvc_tee_layerid, VISIBILITY_OFF, COMMIT_DELAYED);
                }
                vCommitChanges_util();
                rvc_visible = false;
                graphics_visible = false;
                hfp_tee_visible = false;
            }
        }
    }
    ETG_TRACE_FATAL( ( "[%d ms] dispvidctrl shm vHandleVisibilityOnOff() Exited - redrawGuidelinesPending [%d] redrawGammaPending [%d] redrawCSCPending [%d]", OSAL_ClockGetElapsedTime(), redrawGuidelinesPending, redrawGammaPending, redrawCSCPending));
}
#endif //VARIANT_S_FTR_WESTON_FOR_GEN4

// For LSIM - there is no graphics
void display_guidelines(unsigned int width, unsigned int height, unsigned int layer, unsigned int graphics_layer, unsigned int surface, unsigned int rvc_surface, char* buff)
{
   ETG_TRACE_FATAL( ( "[%d ms] dispvidctrl shm display_guidelines() - width [%d], height [%d], layer [%d], graphics layer [%d], surface [%d] rvc surface [%d]", OSAL_ClockGetElapsedTime(), width, height, layer, graphics_layer, surface, rvc_surface));

#if !defined(GEN3X86) && !defined(GEN4LSIM)
   rvc_layerid = layer;
   graphics_layerid = graphics_layer;
   graphics_surfaceid = surface;
   rvc_surfaceid = rvc_surface;
   window_height = height;
   window_width = width;
   guideline_buffer = buff;
   redrawGuidelinesCurrent = 0; //Nothing currently being drawn
   enBlackScreenTypeRequestCurrent = EN_VIDEOPLAYER__BLACKSCREENTYPE_NONE;
   redrawGuidelinesPending = 1; //Set to force a draw
   enBlackScreenTypeRequestPending = EN_VIDEOPLAYER__BLACKSCREENTYPE_NONE;
   display_control = 0;
   stopped = 0;
   running = 1;
   int ret = 0;//lint !e578

#ifdef VARIANT_S_FTR_ENABLE_BUMPER_ICON_P42R//CRQ 844 P42R Bumper Icon
/*surface of the bumper is created here,
 as in the first reverse engage warning text is hid behind bumper is avoided
*/
   if(CreateBumperSurface)
   	display_bumper_graphic();
#endif//VARIANT_S_FTR_ENABLE_BUMPER_ICON_P42R

#ifndef VARIANT_S_FTR_WESTON_FOR_GEN4
   display = create_display();
   window = create_window(display, window_width, window_height);
   if (!window)
   {
      ETG_TRACE_FATAL( ( "[%d ms] dispvidctrl shm display_guidelines() - window could not be created", OSAL_ClockGetElapsedTime()));
      return;
   }
   init_lm_client(window,graphics_layerid,graphics_surfaceid);

   /* Initialise damage to full surface, so the padding gets painted */
   wl_surface_damage(window->surface, 0, 0, window->width, window->height);

   //frame callback has to be deployed before surface_commit(in redraw function)
   if (window->callback)
   {
      wl_callback_destroy(window->callback);
   }

   window->callback = wl_surface_frame(window->surface);
   wl_callback_add_listener(window->callback, &frame_listener, window);

   ETG_TRACE_FATAL( ( "[%d ms] dispvidctrl shm display_guidelines() - graphics ready", OSAL_ClockGetElapsedTime()));
   shm_graphics_ready = true; //graphics has been initialised and ready to display buffer

   redraw(window);

/*Here the RCTA windows has to be created.*/
#ifdef VARIANT_S_FTR_ENABLE_AIVI_RCTA
   display_rcta_graphic();
#endif
   while (running && ret != -1)
   {
      ret = wl_display_dispatch(display->display);

      // wl_display_dispatch exits after surface drawn and callback has returned
           
	 ETG_TRACE_FATAL( ( "[%d ms] dispvidctrl shm display_guidelines() - wl_display_dispatch() exited - redrawGuidelinesPending [%d]", OSAL_ClockGetElapsedTime(), redrawGuidelinesPending));

      //Perform LayerManagerControl commands after surface has been updated
      if(enBlackScreenTypeRequestCurrent != EN_VIDEOPLAYER__BLACKSCREENTYPE_NONE)
      {
         enBlackScreenTypeRequestCurrent = EN_VIDEOPLAYER__BLACKSCREENTYPE_NONE; 
         redrawGuidelinesCurrent = 0;

         // make it visible only if RVC has not stopped meanwhile
         if ( stop_guideline_visibility == false )
         {
            ETG_TRACE_FATAL( ( "[%d ms] dispvidctrl shm display_guidelines surface_visibility = %d", OSAL_ClockGetElapsedTime(),surface_visibility));
            if(surface_visibility == EN_SURFACE_VISIBILITY_SET)
            {
#ifdef VARIANT_S_FTR_ENABLE_BUMPER_ICON_P42R//CRQ 844 P42R Bumper Icon
                set_bumper_surface_status(VISIBILITY_ON);
#endif//VARIANT_S_FTR_ENABLE_BUMPER_ICON_P42R
                bLayerAddSurface(graphics_layerid, graphics_surfaceid, COMMIT_DELAYED);
#ifdef VARIANT_S_FTR_ENABLE_AIVI_NPIVI
                if(rvc_active == true)
#endif
                {
                   vSurfaceSetVisibility(graphics_surfaceid, VISIBILITY_ON, COMMIT_DELAYED);
                   vLayerSetVisibility(graphics_layerid, VISIBILITY_ON, COMMIT_DELAYED);
                   vLayerSetVisibility(rvc_layerid, VISIBILITY_ON, COMMIT_DELAYED);
                   ETG_TRACE_FATAL( ( "[%d ms] dispvidctrl shm display_guidelines() - black screen draw completed - switch on RVC and guideline visibility - ilmUtil", OSAL_ClockGetElapsedTime() ));
                }
#ifdef VARIANT_S_FTR_ENABLE_AIVI_NPIVI
                else {
                   vLayerSetVisibility(graphics_layerid, VISIBILITY_OFF, COMMIT_DELAYED);
                }
#endif //VARIANT_S_FTR_ENABLE_AIVI_NPIVI
                rvc_visible = true;
                graphics_visible = true;
                visibility_changed = true;
            }
            else{
                ETG_TRACE_FATAL( ( "[%d ms] dispvidctrl shm display_guidelines() - wl_display_dispatch() exited ", OSAL_ClockGetElapsedTime()));
            }

#ifdef VARIANT_S_FTR_ENABLE_AIVI_RCTA
            ETG_TRACE_FATAL( ( "[%d ms] dispvidctrl shm display_guidelines() - rcta graphics set", OSAL_ClockGetElapsedTime() ));
            set_rcta_surface_status(VISIBILITY_ON);
#endif //VARIANT_S_FTR_ENABLE_AIVI_RCTA

            // make it visible only if hfp dual window is enabled
            if(hfp_tee_active == true)
            {
               // make tee visible only for AVM Maneuver screen and not for AVM settings screen
               // Do not make it visible if it is already visible
               if( (hfp_tee_visible == false) && (hfp_tee_surfaceactive == true) )
               {
                  ETG_TRACE_FATAL( ( "[%d ms] dispvidctrl shm display_guidelines() - black screen draw completed - switch on tee visibility - ilmUtil", OSAL_ClockGetElapsedTime() ));
                  vSurfaceSetVisibility(rvc_tee_surfaceid, VISIBILITY_ON, COMMIT_DELAYED);
                  vLayerSetVisibility(rvc_tee_layerid, VISIBILITY_ON, COMMIT_DELAYED);
                  hfp_tee_visible = true;
                  visibility_changed = true;
               }
            }
		 

            //Commit the changes only if surface/Layer visibility has changed.
            if(visibility_changed == true)
            {
               vCommitChanges_util();
               visibility_changed = false;
            }
         }

         else 
         {
            //stop_guideline_visibility = false;
            ETG_TRACE_FATAL (("[%d ms] dispvidctrl shm display_guidelines() - RVC has stopped meanwhile, don't make the Layers visible", OSAL_ClockGetElapsedTime()));
         }
      }
      else
      {
         // make it visible only if RVC has not stopped meanwhile
         if ( (stop_guideline_visibility == false) && (rvc_active == true) )
         {
            if (video_signal_error == false)
            {
               // No video signal error. Only switch on visibility if not already visible
               if ( (rvc_visible == false) || (graphics_visible == false) )
               {
                   // There is no current video signal error so we can switch on both video and guidelines
                   ETG_TRACE_FATAL( ( "[%d ms] dispvidctrl shm display_guidelines() - switch on RVC and guideline visibility - ilmUtil", OSAL_ClockGetElapsedTime() ));
#ifdef VARIANT_S_FTR_ENABLE_BUMPER_ICON_P42R//CRQ 844 P42R Bumper Icon
                   set_bumper_surface_status(VISIBILITY_ON);
#endif//VARIANT_S_FTR_ENABLE_BUMPER_ICON_P42R/
                   bLayerAddSurface(graphics_layerid, graphics_surfaceid, COMMIT_DELAYED);
                   vSurfaceSetVisibility(graphics_surfaceid, VISIBILITY_ON, COMMIT_DELAYED);


#ifdef VARIANT_S_FTR_ENABLE_AIVI_RCTA
                   ETG_TRACE_FATAL( ( "[%d ms] dispvidctrl shm display_guidelines() - rcta graphics set", OSAL_ClockGetElapsedTime() ));
                   set_rcta_surface_status(VISIBILITY_ON);
#endif //VARIANT_S_FTR_ENABLE_AIVI_RCTA


                   vLayerSetVisibility(graphics_layerid, VISIBILITY_ON, COMMIT_DELAYED);
                   vLayerSetVisibility(rvc_layerid, VISIBILITY_ON, COMMIT_DELAYED);
                   rvc_visible = true;
                   graphics_visible = true;
                   visibility_changed = true;
               }
               // make it visible only if hfp dual window is enabled
               if(hfp_tee_active == true)
               {
                  // make tee visible only for AVM Maneuver screen and not for AVM settings screen
                  // Do not make it visible if it is already visible
                  if( (hfp_tee_visible == false) && (hfp_tee_surfaceactive == true) )
                  {
                      ETG_TRACE_FATAL( ( "[%d ms] dispvidctrl shm display_guidelines() - black screen draw completed - switch on tee visibility - ilmUtil", OSAL_ClockGetElapsedTime() ));
                      vSurfaceSetVisibility(rvc_tee_surfaceid, VISIBILITY_ON, COMMIT_DELAYED);
                      vLayerSetVisibility(rvc_tee_layerid, VISIBILITY_ON, COMMIT_DELAYED);
                      hfp_tee_visible = true;
                      visibility_changed = true;
                  }
               }
             
               //Commit the changes only if surface/Layer visibility has changed.
               if(visibility_changed == true)
               {
                  vCommitChanges_util();
                  visibility_changed = false;
               }
            }
            else
            {
               // There is currently a video signal error so we can switch on only guidelines
               // Do not switch on if already visible
               if (graphics_visible == false)
               {
                   ETG_TRACE_FATAL( ( "[%d ms] dispvidctrl shm display_guidelines() - current video signal error - switch on only guideline visibility - ilmUtil", OSAL_ClockGetElapsedTime() ));
#ifdef VARIANT_S_FTR_ENABLE_BUMPER_ICON_P42R//CRQ 844 P42R Bumper Icon
                   set_bumper_surface_status(VISIBILITY_ON);
#endif//VARIANT_S_FTR_ENABLE_BUMPER_ICON_P42R
                   bLayerAddSurface(graphics_layerid, graphics_surfaceid, COMMIT_DELAYED);
                   vSurfaceSetVisibility(graphics_surfaceid, VISIBILITY_ON, COMMIT_DELAYED);
#ifdef VARIANT_S_FTR_ENABLE_AIVI_RCTA
                   ETG_TRACE_FATAL( ( "[%d ms] dispvidctrl shm display_guidelines() - rcta graphics set", OSAL_ClockGetElapsedTime() ));
                   set_rcta_surface_status(VISIBILITY_ON);
#endif //VARIANT_S_FTR_ENABLE_AIVI_RCTA
                   vLayerSetVisibility(graphics_layerid, VISIBILITY_ON, COMMIT_DELAYED);
                   vCommitChanges_util();
                   graphics_visible = true;
               }
            }
         }
         else
         {
            //stop_guideline_visibility = false;
            ETG_TRACE_FATAL (("[%d ms] dispvidctrl shm display_guidelines() - RVC has stopped (stop_guideline_visibility), don't make the Layers visible", OSAL_ClockGetElapsedTime()));
         }
      }
      
      // Wait for any update to the Guidelines buffer - redrawGuidelinesPending is set by update_guidelines()
      while(redrawGuidelinesPending == 0)
      {
         // sleep long if camera is not requested to reduce system load (NCG3D-83954)
         if (bSleepLong == true)
         {
            usleep(10000); // 10ms
         }
         else
         {
            usleep(1000); // 1ms
         }

         if ( stop_guideline_visibility == true ) // RVC stopped meanwhile ?
         {
            if ( (rvc_visible == true) || (graphics_visible == true) )
            {
                ETG_TRACE_FATAL( ( "[%d ms] dispvidctrl shm display_guidelines() - RVC stopping - switch off rvc and guideline visibility - ilmUtil", OSAL_ClockGetElapsedTime() ));
#ifdef VARIANT_S_FTR_ENABLE_BUMPER_ICON_P42R//CRQ 844 P42R Bumper Icon
                 set_bumper_surface_status(VISIBILITY_OFF);
#endif//VARIANT_S_FTR_ENABLE_BUMPER_ICON_P42R

                vLayerRemoveSurface(graphics_layerid, graphics_surfaceid, COMMIT_DELAYED);

#ifdef VARIANT_S_FTR_ENABLE_AIVI_RCTA
                ETG_TRACE_FATAL( ( "[%d ms] dispvidctrl shm display_guidelines() - rcta graphics set", OSAL_ClockGetElapsedTime() ));
                set_rcta_surface_status(VISIBILITY_OFF);
#endif //VARIANT_S_FTR_ENABLE_AIVI_RCTA
                vSurfaceSetVisibility(graphics_surfaceid, VISIBILITY_OFF, COMMIT_DELAYED); //Added as this may

                vLayerSetVisibility(graphics_layerid, VISIBILITY_OFF, COMMIT_DELAYED);
                vLayerSetVisibility(rvc_layerid, VISIBILITY_OFF, COMMIT_DELAYED);

                if((hfp_tee_active == true) && (hfp_tee_visible == true) )
                {
                    ETG_TRACE_FATAL( ( "[%d ms] dispvidctrl shm display_guidelines() - RVC stopping - switch off tee visibility - ilmUtil", OSAL_ClockGetElapsedTime() ));
                    vSurfaceSetVisibility(rvc_tee_surfaceid, VISIBILITY_OFF, COMMIT_DELAYED);
                    vLayerSetVisibility(rvc_tee_layerid, VISIBILITY_OFF, COMMIT_DELAYED);
                }
                vCommitChanges_util();
                rvc_visible = false;
                graphics_visible = false;
                hfp_tee_visible = false;
            }
         }
      }

      ETG_TRACE_FATAL( ( "[%d ms] dispvidctrl shm display_guidelines() - redrawGuidelinesPending [%d] - perform update", OSAL_ClockGetElapsedTime(), redrawGuidelinesPending));
      //Perform the update
      //frame callback has to be deployed before surface_commit(in redraw function)
      if (window->callback)
      {
         wl_callback_destroy(window->callback);
      }

      window->callback = wl_surface_frame(window->surface);
      wl_callback_add_listener(window->callback, &frame_listener, window);

      redraw(window);
   }

   ETG_TRACE_FATAL( ( "[%d ms] dispvidctrl shm display_guidelines() - exiting", OSAL_ClockGetElapsedTime()));

   deinit_lm_client(window);
   destroy_window(window);
   destroy_display(display);
   stopped = 1;
//CRQ 844 P42R Bumper Icon
#ifdef VARIANT_S_FTR_ENABLE_BUMPER_ICON_P42R
   remove_bumper_graphics();
#endif//VARIANT_S_FTR_ENABLE_BUMPER_ICON_P42R
#else //VARIANT_S_FTR_WESTON_FOR_GEN4

#ifdef VARIANT_S_FTR_ENABLE_AIVI_PIVI2
   //TBD - need to check why this is necessary
   ETG_TRACE_FATAL( ( "[%d ms] dispvidctrl shm display_guidelines() - PIVI2 setting window size to 1920 x 720", OSAL_ClockGetElapsedTime()));
   window_width = 1920;
   window_height = 720;
#endif //VARIANT_S_FTR_ENABLE_AIVI_PIVI2

  ETG_TRACE_FATAL( ( "[%d ms] dispvidctrl cairo display_guidelines() - calling wayland_init() layer [%d] surface [%d] width [%d] height [%d] buffer [%d]", OSAL_ClockGetElapsedTime(), graphics_layerid, graphics_surfaceid, window_width, window_height, guideline_buffer));
   wayland_init(graphics_layerid, graphics_surfaceid, window_width, window_height, guideline_buffer);
   ETG_TRACE_FATAL( ( "[%d ms] dispvidctrl shm display_guidelines() - cairo - returned from wayland_init()", OSAL_ClockGetElapsedTime()));

#endif //VARIANT_S_FTR_WESTON_FOR_GEN4

#endif //GEN3X86 GEN4LSIM
}


//#######################################################################################################################################################################################################
//#######################################################################################################################################################################################################
// FUNCTIONS CONTAINING INTERFACE TO CONTROL GRAPHICS COMPONENT
//#######################################################################################################################################################################################################
//#######################################################################################################################################################################################################

//******************************************************************************

//FUNCTION:    bool bGraphicsSurfaceReady(void)

//Returns TRUE if graphics has been initialised and is ready to display buffer

//!

//PARAMETER:   None

//RETURNVALUE: bool - TRUE if graphics initialised

//******************************************************************************
bool bGraphicsSurfaceReady(void)
{
   return shm_graphics_ready;
}

#ifdef VARIANT_S_FTR_WESTON_FOR_GEN4
//FUNCTION:    void vSetGraphicsSurfaceReady(void)

//Sets flag to indicate that graphics surface is ready for drawing

//!

//PARAMETER:   None

//RETURNVALUE: None

//******************************************************************************
void vSetGraphicsSurfaceReady(void)
{
   shm_graphics_ready = true;
}

//FUNCTION:    void vSetGammaCSC(void)

//Sets CSC / gamma parameters recieved by dimming component 

//!

//PARAMETER:   None

//RETURNVALUE: None

//******************************************************************************
void vSetGammaCSC(void)
{
	#if !defined(GEN3X86) && !defined(GEN4LSIM)
	redrawGammaPending = 1;
	redrawCSCPending = 1;
    vSetGammaCSC_wayland();
	#endif
}
#endif //VARIANT_S_FTR_WESTON_FOR_GEN4

//******************************************************************************

//FUNCTION:    void set_green_screen(void)

//For LVDS we show a green screen if there is no signal. This switches it on.

//!

//PARAMETER:   None

//RETURNVALUE: None

//******************************************************************************
void set_green_screen(void)
{
   // For LVDS signal failure only
   // We need to show a green screen
   ETG_TRACE_FATAL( ( "[%d ms] dispvidctrl shm set_green_screen()", OSAL_ClockGetElapsedTime()));
   green_screen = true;
   video_signal_error = true;
}

//******************************************************************************

//FUNCTION:    void set_blue_screen(void)

//For CVBS we show a blue screen if there is no signal. This switches it on.

//!

//PARAMETER:   None

//RETURNVALUE: None

//******************************************************************************
void set_blue_screen(void)
{
   // For CVBS signal failure only
   // We need to show a blue screen
   ETG_TRACE_FATAL( ( "[%d ms] dispvidctrl shm set_blue_screen()", OSAL_ClockGetElapsedTime()));
   blue_screen = true;
   video_signal_error = true;
}

//******************************************************************************

//FUNCTION:    void clear_green_screen(void)

//For LVDS we show a green screen if there is no signal. This switches it off.

//!

//PARAMETER:   None

//RETURNVALUE: None

//******************************************************************************
void clear_green_screen(void)
{
   // For LVDS signal restore only
   // We no longer need to show a green screen
   ETG_TRACE_FATAL( ( "[%d ms] dispvidctrl shm clear_green_screen()", OSAL_ClockGetElapsedTime()));
   green_screen = false;
   video_signal_error = false;
}

//******************************************************************************

//FUNCTION:    void clear_blue_screen(void)

//For CVBS we show a blue screen if there is no signal. This switches it off.

//!

//PARAMETER:   None

//RETURNVALUE: None

//******************************************************************************
void clear_blue_screen(void)
{
   // For CVBS signal restore only
   // We no longer need to show a blue screen
   ETG_TRACE_FATAL( ( "[%d ms] dispvidctrl shm clear_blue_screen()", OSAL_ClockGetElapsedTime()));
   blue_screen = false;
   video_signal_error = false;
}

void draw_guidelines(void *image)
{
   tU32 u32TStamp = OSAL_ClockGetElapsedTime();
   ETG_TRACE_FATAL( ( "[%d ms] dispvidctrl shm draw_guidelines() entered - black screen type=%d, blue_screen=%d, green_screen=%d",
                     u32TStamp, ETG_CENUM(ten_VideoPlayer_BlackScreenType, enBlackScreenTypeRequestCurrent), blue_screen, green_screen) );

   if(enBlackScreenTypeRequestCurrent == EN_VIDEOPLAYER__BLACKSCREENTYPE_NONE)
   {
      ETG_TRACE_FATAL( ( "[%d ms] dispvidctrl shm draw_guidelines() - display guidelines", u32TStamp) );
      memcpy(image,guideline_buffer,(window_height * window_width * 4));
   }
   else
   {
      draw_screenmask(image, window_width, window_height, enBlackScreenTypeRequestCurrent);
   }
}

void draw_screenmask(void *image, unsigned int imgwidth, unsigned int imgheight, enum ten_VideoPlayer_BlackScreenType enBlackScreenType)
{
    ETG_TRACE_FATAL(( "[%d ms] dispvidctrl shm draw_screenmask() entered - screen mask type %u", OSAL_ClockGetElapsedTime(), ETG_CENUM(ten_VideoPlayer_BlackScreenType, enBlackScreenType) ));

    // some "dynamic colors" which depend on video status and connection type
    tU32  u32BlackOrVidErr;
    if      (blue_screen  == true) u32BlackOrVidErr  = U32_RGBA_BLUE;
    else if (green_screen == true) u32BlackOrVidErr  = U32_RGBA_GREEN;
    else                           u32BlackOrVidErr  = U32_RGBA_BLACK;

    tU32  u32GrayOrVidErr;
    if      (blue_screen  == true) u32GrayOrVidErr   = U32_RGBA_BLUE;
    else if (green_screen == true) u32GrayOrVidErr   = U32_RGBA_GREEN;
    else                           u32GrayOrVidErr   = U32_RGBA_GRAY;
    
    tU32  u32TranspOrVidErr;
    if      (blue_screen  == true) u32TranspOrVidErr = U32_RGBA_BLUE;
    else if (green_screen == true) u32TranspOrVidErr = U32_RGBA_GREEN;
    else                           u32TranspOrVidErr = U32_RGBA_TRANSP;
    
    tU32* pu32Pix = (tU32*)image;  // initialise the pixel pointer (the VP_SHM_PUT_...() macros below will advance it through the buffer)
    tBool bUnsuppBlkSrcType = false;
    tBool bUnsuppWinSize = false;
    switch (enBlackScreenType)
    {
       case EN_VIDEOPLAYER__BLACKSCREENTYPE_FULL:
       case EN_VIDEOPLAYER__BLACKSCREENTYPE_GAMMA:
       case EN_VIDEOPLAYER__BLACKSCREENTYPE_CSC:
          VP_SHM_PUT_1COLOR_ROWS(pu32Pix, window_height, u32BlackOrVidErr);                  // full window black or blue/green
          break;
       case EN_VIDEOPLAYER__BLACKSCREENTYPE_AVMPHASE3_LHD:  // different (mirrored) masks for LHD and RHD
       case EN_VIDEOPLAYER__BLACKSCREENTYPE_AVMPHASE3_RHD:  // different (mirrored) masks for LHD and RHD
          if (VP_SHM_WINSIZE_IS(1280,768)) {
             VP_SHM_PUT_1COLOR_ROWS(pu32Pix, 128,     u32TranspOrVidErr);                      // 128 transparent or blue/green rows
             if (EN_VIDEOPLAYER__BLACKSCREENTYPE_AVMPHASE3_LHD == enBlackScreenType) {
                VP_SHM_PUT_2COLOR_ROWS(pu32Pix, 533, 781,u32BlackOrVidErr,  499,u32TranspOrVidErr);  // 533 rows of 781 black + 499 transparent pixels, or all blue/green
             } else {
                VP_SHM_PUT_2COLOR_ROWS(pu32Pix, 533, 499,u32TranspOrVidErr, 781,u32BlackOrVidErr );  // 533 rows of 499 transparent + 781 black pixels, or all blue/green
             }
             VP_SHM_PUT_1COLOR_ROWS(pu32Pix, 107,     u32TranspOrVidErr);                      // 107 transparent or blue/green rows
          } else {
             bUnsuppWinSize = true;
          }
          break;
       case EN_VIDEOPLAYER__BLACKSCREENTYPE_AVMPHASE3_WIDE_LHD:  // same (symmetrical) mask for LHD and RHD
       case EN_VIDEOPLAYER__BLACKSCREENTYPE_AVMPHASE3_WIDE_RHD:  // same (symmetrical) mask for LHD and RHD
          if (VP_SHM_WINSIZE_IS(1280,768)) {
             VP_SHM_PUT_1COLOR_ROWS(pu32Pix,  71, u32TranspOrVidErr);                          //  71 transparent or blue/green rows
             VP_SHM_PUT_1COLOR_ROWS(pu32Pix, 590, u32BlackOrVidErr );                          // 590 black or blue/green rows
             VP_SHM_PUT_1COLOR_ROWS(pu32Pix, 107, u32TranspOrVidErr);                          // 107 transparent or blue/green rows
          } else {
             bUnsuppWinSize = true;
          }
          break;
       case EN_VIDEOPLAYER__BLACKSCREENTYPE_RVC_P_LHD:  // same (symmetrical) mask for LHD and RHD
       case EN_VIDEOPLAYER__BLACKSCREENTYPE_RVC_P_RHD:  // same (symmetrical) mask for LHD and RHD
          if (VP_SHM_WINSIZE_IS(820,1024)) {
             VP_SHM_PUT_1COLOR_ROWS(pu32Pix, 335, U32_RGBA_BLACK  );                         // 335 black rows
             VP_SHM_PUT_1COLOR_ROWS(pu32Pix, 514, u32TranspOrVidErr);                        // 514 transparent or blue/green rows
             VP_SHM_PUT_1COLOR_ROWS(pu32Pix, 175, U32_RGBA_BLACK  );                         // 175 black rows
          } else {
             bUnsuppWinSize = true;
          }
          break;
       case EN_VIDEOPLAYER__BLACKSCREENTYPE_RVC_L_LHD:
       case EN_VIDEOPLAYER__BLACKSCREENTYPE_RVC_L_RHD:
          bUnsuppBlkSrcType = true;
          break;
       case EN_VIDEOPLAYER__BLACKSCREENTYPE_AVM_P_LHD:  // same (symmetrical) mask for LHD and RHD
       case EN_VIDEOPLAYER__BLACKSCREENTYPE_AVM_P_RHD:  // same (symmetrical) mask for LHD and RHD
          if (VP_SHM_WINSIZE_IS(820,1024)) {
             VP_SHM_PUT_1COLOR_ROWS(pu32Pix, 205, U32_RGBA_BLACK);                           // 205 black rows
             VP_SHM_PUT_3COLOR_ROWS(pu32Pix, 489, 10,U32_RGBA_BLACK, 800,u32TranspOrVidErr, 10,U32_RGBA_BLACK);  // 489 rows of 10 black + 800 gray/blue/green + 10 black pixels
             VP_SHM_PUT_1COLOR_ROWS(pu32Pix, 330, U32_RGBA_BLACK);                           // 330 black rows
          } else {
             bUnsuppWinSize = true;
          }
          break;
       case EN_VIDEOPLAYER__BLACKSCREENTYPE_AVM_L_LHD:  // same (symmetrical) mask for LHD and RHD
       case EN_VIDEOPLAYER__BLACKSCREENTYPE_AVM_L_RHD:  // same (symmetrical) mask for LHD and RHD
          if (VP_SHM_WINSIZE_IS(800,480)) {
             VP_SHM_PUT_1COLOR_ROWS(pu32Pix, window_height, u32GrayOrVidErr);                // full window gray or blue/green
          } else {
             bUnsuppWinSize = true;
          }
          break;
       case EN_VIDEOPLAYER__BLACKSCREENTYPE_MVC_L_LHD:  // different (mirrored) masks for LHD and RHD
       case EN_VIDEOPLAYER__BLACKSCREENTYPE_MVC_L_RHD:  // different (mirrored) masks for LHD and RHD
          if (VP_SHM_WINSIZE_IS(800,480)) {
             if (EN_VIDEOPLAYER__BLACKSCREENTYPE_MVC_L_LHD == enBlackScreenType) {
                VP_SHM_PUT_2COLOR_ROWS(pu32Pix, 402,  90,U32_RGBA_BLACK,   710,u32TranspOrVidErr); // 402 rows of 90 black + 710 transparent/blue/green pixels
             } else {
                VP_SHM_PUT_2COLOR_ROWS(pu32Pix, 402, 710,u32TranspOrVidErr, 90,U32_RGBA_BLACK);    // 402 rows of 710 transparent/blue/green + 90 black pixels
             }
             VP_SHM_PUT_1COLOR_ROWS(pu32Pix,  78,     U32_RGBA_BLACK);                       // 78 black rows (zone 3)
          } else {
             bUnsuppWinSize = true;
          }
          break;
       case EN_VIDEOPLAYER__BLACKSCREENTYPE_AVM_HFP_P_LHD:  // same (symmetrical) mask for LHD and RHD
       case EN_VIDEOPLAYER__BLACKSCREENTYPE_AVM_HFP_P_RHD:  // same (symmetrical) mask for LHD and RHD
           if (VP_SHM_WINSIZE_IS(820,1024)) {
              //NCG3D-216308 HFP in Guidance mode activate AVM where display is not proper
              // VP_SHM_PUT_1COLOR_ROWS(pu32Pix, 335, U32_RGBA_BLACK);                           // 335 black rows
              if (EN_VIDEOPLAYER__BLACKSCREENTYPE_AVM_HFP_P_LHD == enBlackScreenType) {
                 VP_SHM_PUT_2COLOR_ROWS(pu32Pix, 335,  649,U32_RGBA_BLACK,   171,u32TranspOrVidErr); // 335 rows of 649 black + 171 transparent/blue/green pixels
              } else {
                 VP_SHM_PUT_2COLOR_ROWS(pu32Pix, 335, 171,u32TranspOrVidErr, 649,U32_RGBA_BLACK);    // 335 rows of 171 transparent/blue/green + 649 black pixels
              }
               VP_SHM_PUT_1COLOR_ROWS(pu32Pix, 514, u32TranspOrVidErr);                        // 514 transparent/blue/green rows
               VP_SHM_PUT_1COLOR_ROWS(pu32Pix, 175, U32_RGBA_BLACK);                           // 175 black rows
           } else {
              bUnsuppWinSize = true;
           }
           break;
       case EN_VIDEOPLAYER__BLACKSCREENTYPE_AVM_SETTINGS_P_LHD:  // same (symmetrical) mask for LHD and RHD
       case EN_VIDEOPLAYER__BLACKSCREENTYPE_AVM_SETTINGS_P_RHD:  // same (symmetrical) mask for LHD and RHD
          if (VP_SHM_WINSIZE_IS(820,1024)) {
             VP_SHM_PUT_1COLOR_ROWS(pu32Pix, 136, U32_RGBA_BLACK);                           // 136 black rows
             VP_SHM_PUT_3COLOR_ROWS(pu32Pix, 461, 30,U32_RGBA_BLACK, 760,u32TranspOrVidErr, 30,U32_RGBA_BLACK);  // 461 rows of 30 black + 760 transparent/blue/green + 30 black pixels
             VP_SHM_PUT_1COLOR_ROWS(pu32Pix, 427, U32_RGBA_BLACK);                           // 427 black rows
          } else {
             bUnsuppWinSize = true;
          }
          break;
       case EN_VIDEOPLAYER__BLACKSCREENTYPE_NONE:
          VP_SHM_PUT_1COLOR_ROWS(pu32Pix, window_height, U32_RGBA_TRANSP);                  // full window transparent
          break;
       default:
          bUnsuppBlkSrcType = true;
          break;
    }
    
    if (bUnsuppBlkSrcType) {
       ETG_TRACE_FATAL( ( "[%d ms] dispvidctrl shm draw_screenmask() - requested black screen type %d not handled/supported",
                         OSAL_ClockGetElapsedTime(), ETG_CENUM(ten_VideoPlayer_BlackScreenType, enBlackScreenType)) );
    } else if (bUnsuppWinSize) {
       ETG_TRACE_FATAL( ( "[%d ms] dispvidctrl shm draw_screenmask() - current window size %d x %d not handled/supported for black screen type %d",
                         OSAL_ClockGetElapsedTime(), window_width,window_height, ETG_CENUM(ten_VideoPlayer_BlackScreenType, enBlackScreenType)) );
    }
}



// For LSIM - there is no graphics
void update_guidelines(enum ten_VideoPlayer_BlackScreenType enBlackScreenType)
{
#if !defined(GEN3X86) && !defined(GEN4LSIM)
   ETG_TRACE_FATAL( ( "[%d ms] dispvidctrl shm update_guidelines() - entered - enBlackScreenType [%d], running [%d]", OSAL_ClockGetElapsedTime(), ETG_CENUM(ten_VideoPlayer_BlackScreenType, enBlackScreenType), running));
   if(running == 1)
   {
      ETG_TRACE_FATAL( ( "[%d ms] dispvidctrl shm update_guidelines() - currently displaying graphics - setting redrawGuidelinesPending = 1 to force update", OSAL_ClockGetElapsedTime()));
      //stop_guideline_visibility = false;
      redrawGuidelinesPending = 1; // Set as pending in case a previous update is ongoing

      if(force_Blackscreen == TRUE)
      {
         enBlackScreenTypeRequestPending = EN_VIDEOPLAYER__BLACKSCREENTYPE_FULL;  //Fix for NCG3D-110200
      }
      else
      {
         enBlackScreenTypeRequestPending = enBlackScreenType;
      }
   }
   else
   {
      ETG_TRACE_FATAL( ( "[%d ms] dispvidctrl shm update_guidelines() - not currently displaying graphics - ignoring", OSAL_ClockGetElapsedTime()));
   }
#endif //GEN3X86 GEN4LSIM
}

//To update black screen forcefully
void update_Blackscreen(bool TailgateOpen, bool ReverseActive)
{
    ETG_TRACE_FATAL( ( "dispvidctrl shm update_Blackscreen() - setting Black screen force update TailgateOpen = %d ReverseActive = %d", TailgateOpen, ReverseActive));

    if(TailgateOpen && ReverseActive)
    {
        force_Blackscreen = TRUE;  // Update black screen only when reverse is engaged and tailgate is opened. Fix for NCG3D-110200.
    }
    else
    {
        force_Blackscreen = FALSE;  //Otherwise don't force update black screen.
    }
}

// For LSIM - there is no graphics
void end_display_guidelines()
{
#if !defined(GEN3X86) && !defined(GEN4LSIM)
   stop_guideline_visibility = true;
   green_screen = false;
   blue_screen = false;
   video_signal_error = false;
   bSleepLong = true; // thread can sleep for longer periods to reduce system load
   hfp_tee_visible = false;
   ETG_TRACE_FATAL( ( "[%d ms] dispvidctrl shm  end_display_guidelines entered &  stop_guideline_visibility=%d", OSAL_ClockGetElapsedTime(),stop_guideline_visibility));
#endif //GEN3X86 GEN4LSIM
}

void stop_graphics_complete()
{
   stop_guideline_visibility = false;
   ETG_TRACE_FATAL( ( "[%d ms] dispvidctrl shm  stop_graphics_complete() entered & stop_guideline_visibility=%d", OSAL_ClockGetElapsedTime(),stop_guideline_visibility));
}

void set_rvc_active(bool bRVCActive)
{
   ETG_TRACE_FATAL( ( "[%d ms] dispvidctrl shm set_rvc_active() bRVCActive [%d]", OSAL_ClockGetElapsedTime(), bRVCActive));
   rvc_active = bRVCActive;
}

void set_rvc_visible(bool bRVCVisible)
{
   ETG_TRACE_FATAL( ( "[%d ms] dispvidctrl shm set_rvc_visible() bRVCVisible [%d]", OSAL_ClockGetElapsedTime(), bRVCVisible));
   rvc_visible = bRVCVisible;
}

void set_graphics_visible(bool bGraphicsVisible)
{
   ETG_TRACE_FATAL( ( "[%d ms] dispvidctrl shm set_graphics_visible() bGraphicsVisible [%d]", OSAL_ClockGetElapsedTime(), bGraphicsVisible));
   graphics_visible = bGraphicsVisible;
}

void set_hfp_tee_active(unsigned int teeLayerId, unsigned int teeSurfaceId)
{
   hfp_tee_active = true;
   rvc_tee_layerid = teeLayerId;
   rvc_tee_surfaceid = teeSurfaceId;
}

void set_hfp_tee_inactive(void)
{
   hfp_tee_active = false;
#ifdef VARIANT_S_FTR_ENABLE_AIVI_SCOPE2_1
//NCG3D-228054 rvp4kor issue seen for Renault the surface 89 visibility was not set hfp_tee_active was set to true
   hfp_tee_visible = false;
#endif
}

//Set the TEE surface visibility based on videostate - AVM Maneuver screen/Settings Screen.
void set_tee_surface_visibility(bool hpf_tee_surface)
{
   hfp_tee_surfaceactive = hpf_tee_surface;
   ETG_TRACE_FATAL( ( "[%d ms] dispvidctrl shm set_hfp_tee_window() - hfp_tee_windowactive = %d", OSAL_ClockGetElapsedTime(), hfp_tee_surfaceactive));
   if(hfp_tee_surfaceactive == false)
   {
      hfp_tee_visible = false;  //Set the tee surface visibility to false, as tee window should not be displayed in AVM settings screen.
   }
   ETG_TRACE_FATAL( ( "[%d ms] dispvidctrl shm set_hfp_tee_window() - hfp_tee_visible = %d", OSAL_ClockGetElapsedTime(), hfp_tee_visible));
}


void reset_ThreadSleepLong(void)
{
   bSleepLong = false;
}
bool bGetThreadSleepLongState(void)
{
   return (bSleepLong);
}

//******************************************************************************

//FUNCTION:    void stop_pending_update(void)

//! Stop any further pending update to the Graphics surface. Used by vStop().

//!

//PARAMETER: None

//RETURNVALUE: None

//******************************************************************************
void stop_pending_update(void)
{
   ETG_TRACE_FATAL( ( "[%d ms] dispvidctrl shm stop_pending_update() - redrawGuidelinesPending reset", OSAL_ClockGetElapsedTime()));
   redrawGuidelinesPending = 0;
   ETG_TRACE_FATAL( ( "[%d ms] dispvidctrl shm stop_pending_update() called with enBlackScreenTypeRequestCurrent= %d", OSAL_ClockGetElapsedTime(),enBlackScreenTypeRequestCurrent));
   enBlackScreenTypeRequestCurrent = EN_VIDEOPLAYER__BLACKSCREENTYPE_NONE;
}

void set_surface_visiblity()
{
   surface_visibility = EN_SURFACE_VISIBILITY_SET;
   ETG_TRACE_FATAL( ( "[%d ms] dispvidctrl shm set_surface_visiblity - set surface visibility true -- surface_visibility = %d", OSAL_ClockGetElapsedTime(),surface_visibility));
}

void stop_surface_visiblity()
{
   surface_visibility = EN_SURFACE_DESTROYED;
   ETG_TRACE_FATAL( ( "[%d ms] dispvidctrl shm stop_surface_visiblity() - surface visibility=%d", OSAL_ClockGetElapsedTime(),surface_visibility));
}

#ifdef VARIANT_S_FTR_WESTON_FOR_GEN4

void vSetGamma_shm(double dRGamma_para,
                  double dGGamma_para,
                  double dBGamma_para)
{
    ETG_TRACE_FATAL( ( "[%d ms] dispvidctrl shm vSetGamma_shm() - Gamma [%d, %d, %d]", 
	          OSAL_ClockGetElapsedTime(),
                  dRGamma_para,
                  dGGamma_para,
                  dBGamma_para));
				  
    dRGamma                = dRGamma_para;
    dGGamma                = dGGamma_para;
    dBGamma                = dBGamma_para;
}

void vSetCSC_shm(unsigned int u16CSCHue_para,
                 unsigned int u16CSCSaturation_para,
                 unsigned int u16CSCBrightness_para,
                 unsigned int u16CSCContrast_para,
                 unsigned int u16CSCHueOffset_para,
                 unsigned int u16CSCSaturationOffset_para,
                 unsigned int u16CSCBrightnessOffset_para)
{
    ETG_TRACE_FATAL( ( "[%d ms] dispvidctrl shm vSetCSC_shm() - CSC [%d, %d, %d, %d, %d, %d, %d]", 
                  OSAL_ClockGetElapsedTime(),
                  u16CSCHue_para,
                  u16CSCSaturation_para,
                  u16CSCBrightness_para,
                  u16CSCContrast_para,
                  u16CSCHueOffset_para,
                  u16CSCSaturationOffset_para,
                  u16CSCBrightnessOffset_para));
				  
    u16CSCHue              = u16CSCHue_para;
    u16CSCSaturation       = u16CSCSaturation_para;
    u16CSCBrightness       = u16CSCBrightness_para;
    u16CSCContrast         = u16CSCContrast_para;
    u16CSCHueOffset        = u16CSCHueOffset_para;
    u16CSCSaturationOffset = u16CSCSaturationOffset_para;
    u16CSCBrightnessOffset = u16CSCBrightnessOffset_para;
}

#endif //VARIANT_S_FTR_WESTON_FOR_GEN4

//For creating a new surface for the RCTA changes.................//
#ifdef VARIANT_S_FTR_ENABLE_AIVI_RCTA
/*
	static void my_frame_callback_rcta(void *data, struct wl_callback *callback, uint32_t time)
	{//lint !e578
		 UNUSED(data);
		 UNUSED(callback);
		 UNUSED(time);
		 ETG_TRACE_FATAL( ( "[%d ms] dispvidctrl shm my_frame_callback_rcta() - frame processed", OSAL_ClockGetElapsedTime()));
	}

	static const struct wl_callback_listener frame_listener_rcta = {
		my_frame_callback_rcta
	};
*/
	void rcta_init_graphics(unsigned short x_pos, unsigned short y_pos, unsigned short width, unsigned short height, unsigned short rcta_layerid, unsigned short rcta_surfaceid, char *buff)
	{
		ETG_TRACE_FATAL( ( "[%d ms] dispvidctrl shm rcta_init_graphics ",OSAL_ClockGetElapsedTime()));

		//rcta_window_size = (struct rcta_window_params *)malloc(sizeof(struct rcta_window_params));
		if(buff)
		{
			rcta_window_size.window_X_pos = x_pos;//Default set to zero so as the surface drawn at 0 location
			rcta_window_size.window_Y_pos = y_pos;//Y offset value changes based on Display
			rcta_window_size.window_width = width; //Window width to be the same as the width of the Display
			rcta_window_size.window_height = height; //Window Height will be the maximum value of the Image
			rcta_window_size.rcta_layerid = rcta_layerid;// The layer is same as the Guideline layer
			rcta_window_size.rcta_surfaceid = rcta_surfaceid;// The surface is set to the value 98

			rcta_image_buffer = buff;//Intialize the pointer of the buffer

			rcta_init_done = true;
		}
		else
		{
			ETG_TRACE_FATAL( ( "[%d ms] dispvidctrl shm rcta_init_graphics failed ",OSAL_ClockGetElapsedTime()));
			rcta_init_done = false;
		}
	}

	void display_rcta_graphic()
	{
		//New rcta_window is created for the RCTA
		if(rcta_init_done)
		{
			ETG_TRACE_FATAL( ( "[%d ms] display_rcta_graphic() - width [%d], height [%d], layer [%d], surface [%d] ", OSAL_ClockGetElapsedTime(), rcta_window_size.window_width, rcta_window_size.window_height, rcta_window_size.rcta_layerid, rcta_window_size.rcta_surfaceid));

			if(!stop_rcta_graphic)
			{
				rcta_display = create_display();
				if (!rcta_display)
					return;
				rcta_window = create_window(rcta_display, rcta_window_size.window_width, rcta_window_size.window_height);

				if (!rcta_window)
				{
					ETG_TRACE_FATAL(( "[%d ms] dispvidctrl shm display_rcta_graphic() - rcta_window could not be created", OSAL_ClockGetElapsedTime()));
					return;
				}

				init_lm_client(rcta_window, rcta_window_size.rcta_layerid, rcta_window_size.rcta_surfaceid);
				ETG_TRACE_FATAL( ( "[%d ms] dispvidctrl shm init_lm_client exit() - ", OSAL_ClockGetElapsedTime()));


				/* Initialise damage to full surface, so the padding gets painted */
				wl_surface_damage(rcta_window->surface, 0, 0, rcta_window->width, rcta_window->height);
				//frame callback has to be deployed before surface_commit(in redraw function)

				rcta_graphic_framecallback();

				ETG_TRACE_FATAL( ( "[%d ms] dispvidctrl shm rcta_graphic_framecallback exit() - ", OSAL_ClockGetElapsedTime()));

				//draw_rcta_image(rcta_window);
				if(rcta_image_buffer)
				{
					show_rcta_graphics(rcta_image_buffer);
				}
				//set the destination of the surface to the correct position
				rcta_graphics_ready = TRUE;
				ETG_TRACE_FATAL( ( "[%d ms] dispvidctrl shm display_rcta_graphic() -rcta_graphics_ready", OSAL_ClockGetElapsedTime()));
			}
			else
			{
				ETG_TRACE_FATAL( ( "[%d ms] dispvidctrl shm  failed to create window", OSAL_ClockGetElapsedTime()));
			}
		} else {
			ETG_TRACE_FATAL( ( "[%d ms] display_rcta_graphic() - rcta_init_done is fasle", OSAL_ClockGetElapsedTime()));
		}
	}

	//set the surface and layer visible.
	void set_rcta_surface_status(bool visibility)//fuction name change
	{
		if(rcta_init_done)
		{
			ETG_TRACE_FATAL( ( "dispvidctrl set_rcta_surface_on_off() -  [%d]", visibility));
			if(!stop_rcta_graphic)
			{
				if(visibility)
				{
					bLayerAddSurface(rcta_window_size.rcta_layerid, rcta_window_size.rcta_surfaceid, COMMIT_DELAYED);
					//vSurfaceSetDestinationRectangle(rcta_window_size.rcta_surfaceid, rcta_window_size.window_X_pos, rcta_window_size.window_Y_pos, rcta_window_size.window_width, rcta_window_size.window_height, COMMIT_DELAYED);
				}
				else
				{
					vLayerRemoveSurface(rcta_window_size.rcta_layerid, rcta_window_size.rcta_surfaceid, COMMIT_DELAYED);
				}
				vSurfaceSetVisibility(rcta_window_size.rcta_surfaceid, visibility, COMMIT_DELAYED);
			}
		}
	}


	void show_rcta_graphics(char *buff)
	{
		ETG_TRACE_FATAL(( "dispvidctrl show_rcta()"));
		rcta_image_buffer = buff;//Intialize the pointer of the buffer
		if(rcta_graphics_ready)
		{
			draw_rcta_image(rcta_window);
		}
	}

	void remove_rcta_graphics()
	{
		ETG_TRACE_FATAL(( "dispvidctrl stop_rcta()"));

        set_rcta_surface_status(false);
		stop_rcta_graphic = true;//Intialize the pointer of the buffer
		if(rcta_init_done)
		{
			rcta_init_done = false;
			//if(rcta_window_size != NULL)
			//{
			//	free(rcta_window_size);
			//	rcta_window_size = NULL;
			//}
			rcta_image_buffer = NULL;
		}
		//check if graphics created previously
		if(rcta_graphics_ready)
		{
			destroy_rcta_window();
			rcta_graphics_ready = false;
		}
	}

	void draw_rcta_image(void *data)
	{
		if (!rcta_window)
			return;
		if(rcta_init_done)//This is used as guard condition
		{
			struct window *r_window = data; //lint !e578
			struct buffer *r_buffer;

			ETG_TRACE_FATAL( ( "[%d ms] dispvidctrl shm draw_rcta_image() - Entering - ",OSAL_ClockGetElapsedTime()));
			r_buffer = window_next_buffer(r_window);

			if(!r_buffer)
			{
				ETG_TRACE_FATAL( ( "[%d ms] dispvidctrl shm draw_rcta_image() Failed after retrying create buffer, triggering reset", OSAL_ClockGetElapsedTime()));
				abort();
			}
			
			memcpy(r_buffer->shm_data, rcta_image_buffer, ((rcta_window_size.window_width)*(rcta_window_size.window_height)*4));

			rcta_graphic_framecallback();

			wl_surface_attach(r_window->surface, r_buffer->buffer, 0, 0);
			wl_surface_damage(r_window->surface, 0, 0, r_window->width, r_window->height);
			wl_surface_commit(r_window->surface);
			r_buffer->busy = 1;

			int ret = wl_display_dispatch(rcta_display->display);

			ETG_TRACE_FATAL( ( "[%d ms] dispvidctrl shm draw_rcta_image  wl_display_dispatch() -", OSAL_ClockGetElapsedTime()));
			if(ret == -1)
			{
				remove_rcta_graphics();
			}
			ETG_TRACE_FATAL( ( "[%d ms] dispvidctrl shm draw_rcta_image() - exit - ",OSAL_ClockGetElapsedTime()));
		}
	}

	void rcta_graphic_framecallback()
	{
		if (!rcta_window)
			return;
		 ETG_TRACE_FATAL( ( "[%d ms] dispvidctrl shm rcta_graphic_framecallback() - ", OSAL_ClockGetElapsedTime()));
		//frame callback has to be deployed before surface_commit(in redraw function)
		if (rcta_window->callback)
		{
			wl_callback_destroy(rcta_window->callback);
		}
		rcta_window->callback = wl_surface_frame(rcta_window->surface);
		//wl_callback_add_listener(rcta_window->callback, &frame_listener_rcta, rcta_window);
		ETG_TRACE_FATAL( ( "[%d ms] dispvidctrl shm wl_display_dispatch inside() -", OSAL_ClockGetElapsedTime()));
	}

	//function destroy the created surface and the window.
	void destroy_rcta_window()
	{
		if (!rcta_window)
			return;
		ETG_TRACE_FATAL( ( "[%d ms] dispvidctrl shm destroy_rcta_window() -", OSAL_ClockGetElapsedTime()));
		deinit_lm_client(rcta_window);
		destroy_window(rcta_window);
		destroy_display(rcta_display);
	}

	bool rcta_graphic_status()
	{
		return rcta_graphics_ready;
	}

#endif
//CRQ 844 P42R Bumper Icon
#ifdef VARIANT_S_FTR_ENABLE_BUMPER_ICON_P42R
	/*static void my_frame_callback_bumper(void *data, struct wl_callback *callback, uint32_t time)
	{//lint !e578
		 UNUSED(data);
		 UNUSED(callback);
		 UNUSED(time);
		 ETG_TRACE_FATAL( ( "[%d ms] dispvidctrl shm my_frame_callback_bumper() - frame processed", OSAL_ClockGetElapsedTime()));
	}

	static const struct wl_callback_listener frame_listener_bumper = {
		my_frame_callback_bumper
	};*/

	void bumper_init_graphics(unsigned short x_pos, unsigned short y_pos, unsigned short width, unsigned short height, unsigned short bumper_layerid, unsigned short bumper_surfaceid, char *buff)
	{
	   ETG_TRACE_FATAL( ( "[%d ms] dispvidctrl shm bumper_init_graphics ",OSAL_ClockGetElapsedTime()));
       if(CreateBumperSurface)
       {
          bumper_window_size = (struct bumper_window_params *)malloc(sizeof(struct bumper_window_params));

          if(bumper_window_size)
          {
             bumper_window_size->window_X_pos = x_pos;//Default set to zero so as the surface drawn at 0 location
             bumper_window_size->window_Y_pos = y_pos;//Y offset value changes based on Display
             bumper_window_size->window_width = width; //Window width to be the same as the width of the Display
             bumper_window_size->window_height = height; //Window Height will be the maximum value of the Image
             bumper_window_size->bumper_layerid = bumper_layerid;// The layer is same as the Guideline layer
             bumper_window_size->bumper_surfaceid = bumper_surfaceid;// The surface is set to the value 95

             bumper_image_buffer = buff;//Intialize the pointer of the buffer

             bumper_init_done = true;
          }
          else
          {
             ETG_TRACE_FATAL( ( "[%d ms] dispvidctrl shm bumper_init_graphics failed ",OSAL_ClockGetElapsedTime()));
             CreateBumperSurface = false;
             bumper_init_done = false;
          }
      }
	}

	void display_bumper_graphic()
	{
		//New bumper_window is created for the bumper
		if(CreateBumperSurface)
		{
			if(bumper_init_done)
			{
				ETG_TRACE_FATAL( ( "[%d ms] display_bumper_graphic() - width [%d], height [%d], layer [%d], surface [%d] ", OSAL_ClockGetElapsedTime(), bumper_window_size->window_width, bumper_window_size->window_height, bumper_window_size->bumper_layerid, bumper_window_size->bumper_surfaceid));

				if(!stop_bumper_graphic)
				{
					bumper_display = create_display();
					if (!bumper_display)
					{
						ETG_TRACE_FATAL(( "[%d ms] dispvidctrl shm display_bumper_graphic() - bumper_display could not be created", OSAL_ClockGetElapsedTime()));
						bumper_window = NULL;
						remove_bumper_graphics();
						return;
					}
					bumper_window = create_window(bumper_display, bumper_window_size->window_width, bumper_window_size->window_height);

					if (!bumper_window)
					{
						ETG_TRACE_FATAL(( "[%d ms] dispvidctrl shm display_bumper_graphic() - bumper_window could not be created", OSAL_ClockGetElapsedTime()));
						bumper_display = NULL;
						remove_bumper_graphics();
						return;
					}

					init_lm_client(bumper_window, bumper_window_size->bumper_layerid, bumper_window_size->bumper_surfaceid);
					ETG_TRACE_FATAL( ( "[%d ms] dispvidctrl shm init_lm_client exit() - ", OSAL_ClockGetElapsedTime()));


					/* Initialise damage to full surface, so the padding gets painted */
					wl_surface_damage(bumper_window->surface, 0, 0, bumper_window->width, bumper_window->height);
					//frame callback has to be deployed before surface_commit(in redraw function)

					bumper_graphic_framecallback();

					ETG_TRACE_FATAL( ( "[%d ms] dispvidctrl shm bumper_graphic_framecallback exit() - ", OSAL_ClockGetElapsedTime()));
					bumper_graphics_ready = TRUE;
					if(bumper_image_buffer)
						show_bumper_graphics(bumper_image_buffer);
					//set the destination of the surface to the correct position
					ETG_TRACE_FATAL( ( "[%d ms] dispvidctrl shm display_bumper_graphic() -bumper_graphics_ready", OSAL_ClockGetElapsedTime()));
				}
				else
				{
					ETG_TRACE_FATAL( ( "[%d ms] dispvidctrl shm  failed to create window", OSAL_ClockGetElapsedTime()));
				}
			}
			else
			{
				ETG_TRACE_FATAL( ( "[%d ms] display_bumper_graphic() - bumper_init_done is fasle", OSAL_ClockGetElapsedTime()));
			}
		}
	}

	//set the surface and layer visible.
	void set_bumper_surface_status(bool visibility)//fuction name change
	{
		if (!bumper_window)
		{
			ETG_TRACE_FATAL(( "[%d ms] dispvidctrl shm set_bumper_surface_status() - bumper_display could not be created", OSAL_ClockGetElapsedTime()));
			return;
		}
		if(CreateBumperSurface)
		{
         if(bumper_init_done)
         {
            ETG_TRACE_FATAL( ( "dispvidctrl set_bumper_surface_on_off() -  [%d]", visibility));
            if(!stop_bumper_graphic)
            {
               if(visibility)
               {
                  bLayerAddSurface(bumper_window_size->bumper_layerid, bumper_window_size->bumper_surfaceid, COMMIT_DELAYED);
               }
               else
               {
                  vLayerRemoveSurface(bumper_window_size->bumper_layerid, bumper_window_size->bumper_surfaceid, COMMIT_DELAYED);
               }
               vSurfaceSetVisibility(bumper_window_size->bumper_surfaceid, visibility, COMMIT_DELAYED);
            }
         }
		}
    }


	void show_bumper_graphics(char *buff)
	{
		ETG_TRACE_FATAL(( "dispvidctrl show_bumper_graphics()"));
		if(CreateBumperSurface)
		{
         if(bumper_graphics_ready)
         {
            bumper_image_buffer = buff;//Intialize the pointer of the buffer
			if (!bumper_window)
			{
				ETG_TRACE_FATAL(( "[%d ms] dispvidctrl shm show_bumper_graphics() - bumper_display could not be created", OSAL_ClockGetElapsedTime()));
				return;
			}
            draw_bumper_image(bumper_window);
         }
		}
	}

	void draw_bumper_image(void *data)
	{
		struct window *r_window = data; //lint !e578
		struct buffer *r_buffer;
		if(CreateBumperSurface)
		{
			if(bumper_init_done)
			{
				ETG_TRACE_FATAL( ( "[%d ms] dispvidctrl shm draw_bumper_image() - Entering - ",OSAL_ClockGetElapsedTime()));
				r_buffer = window_next_buffer(r_window);

				if(!r_buffer)
				{
					ETG_TRACE_FATAL( ( "[%d ms] dispvidctrl shm draw_bumper_image() Failed after retrying create buffer, triggering reset", OSAL_ClockGetElapsedTime()));
					abort();
				}

				memcpy(r_buffer->shm_data, bumper_image_buffer, ((bumper_window_size->window_width)*(bumper_window_size->window_height)*4));

				bumper_graphic_framecallback();

				wl_surface_attach(r_window->surface, r_buffer->buffer, 0, 0);
				wl_surface_damage(r_window->surface, 0, 0, r_window->width, r_window->height);
				wl_surface_commit(r_window->surface);
				r_buffer->busy = 1;

				int ret = wl_display_dispatch(bumper_display->display);

				ETG_TRACE_FATAL( ( "[%d ms] dispvidctrl shm draw_bumper_image  wl_display_dispatch() -", OSAL_ClockGetElapsedTime()));
				if(ret == -1)
				{
					destroy_bumper_window();
				}
				ETG_TRACE_FATAL( ( "[%d ms] dispvidctrl shm draw_bumper_image() - exit - ",OSAL_ClockGetElapsedTime()));
			}
		}
	}

	void bumper_graphic_framecallback()
	{
		if (!bumper_window)
		{
			ETG_TRACE_FATAL(( "[%d ms] dispvidctrl shm bumper_graphic_framecallback() - bumper_display could not be created", OSAL_ClockGetElapsedTime()));
			return;
		}
		ETG_TRACE_FATAL( ( "[%d ms] dispvidctrl shm bumper_graphic_framecallback() - ", OSAL_ClockGetElapsedTime()));
		//frame callback has to be deployed before surface_commit(in redraw function)
		if (bumper_window->callback)
		{
			wl_callback_destroy(bumper_window->callback);
		}
		bumper_window->callback = wl_surface_frame(bumper_window->surface);
		//wl_callback_add_listener(bumper_window->callback, &frame_listener_bumper, bumper_window);
		ETG_TRACE_FATAL( ( "[%d ms] dispvidctrl shm wl_display_dispatch inside() -", OSAL_ClockGetElapsedTime()));
	}
//remove bumper
	void remove_bumper_graphics()
	{
		ETG_TRACE_FATAL(( "dispvidctrl remove_bumper_graphics()"));
		set_bumper_surface_status(false);
		CreateBumperSurface = false;
		stop_bumper_graphic = true;//Intialize the pointer of the buffer
		if(bumper_init_done)
		{
			bumper_init_done = false;
			if(bumper_window_size != NULL)
			{
				free(bumper_window_size);
				bumper_window_size = NULL;
			}
			bumper_image_buffer = NULL;
		}
		//check if graphics created previously
		if(bumper_graphics_ready)
		{
			bumper_graphics_ready = false;
			destroy_bumper_window();
		}
	}
	//function destroy the created surface and the window.
	void destroy_bumper_window()
	{
		if (!bumper_window)
		{
			ETG_TRACE_FATAL(( "[%d ms] dispvidctrl shm destroy_bumper_window() - bumper_display could not be created", OSAL_ClockGetElapsedTime()));
			return;
		}
		deinit_lm_client(bumper_window);
		destroy_window(bumper_window);
		if (!bumper_display)
		{
			ETG_TRACE_FATAL(( "[%d ms] dispvidctrl shm destroy_bumper_window() - bumper_display could not be created", OSAL_ClockGetElapsedTime()));
			return;
		}
		destroy_display(bumper_display);
	}

	void set_BumperSurfaceRequired(bool bSetting)
	{
		CreateBumperSurface = bSetting;
	}

	bool get_BumperGraphicStatus()
	{
		return bumper_graphics_ready;
	}
#endif//VARIANT_S_FTR_ENABLE_BUMPER_ICON_P42R
